/*
	Author: Higher Logic
	Purpose: Creates community memberships for use in the Higher Logic platform.
	Remarks: Users are added to communities in HL based ON the GroupKey, so it is important that the GroupKey is unique for each intended community.
*/
CREATE VIEW dbo.hlMemberCommunityGroups
AS
		SELECT
			a.ID AS MemberKey,
			a.CHAPTER AS GroupKey,
			b.TITLE AS GroupName,
			'Chapter' AS GroupType,
			'Member' AS ROLE,
			NULL AS SinceDate,
			NULL AS StartDate,
			a.PAID_THRU AS EndDate
		FROM Name a
			INNER JOIN Product b
			ON a.CHAPTER = b.PRODUCT_MINOR
				AND b.PRODUCT_MAJOR = 'CHAPT'
		WHERE a.PAID_THRU > GetDate()

	UNION
		SELECT
			a.ID AS MemberKey,
			a.OTHER_CODE AS GroupKey,
			a.DESCRIPTION AS GroupName,
			'Committee' AS GroupType,
			CASE
			WHEN Len(RTrim(b.TITLE)) > 0 THEN b.TITLE
			ELSE 'Member'
		END AS ROLE,
			NULL AS SinceDate,
			a.EFFECTIVE_DATE AS StartDate,
			a.THRU_DATE AS EndDate
		FROM Activity a
			LEFT OUTER JOIN Committee_Position b
			ON a.ACTION_CODES = b.POSITION_CODE
		WHERE a.ACTIVITY_TYPE = 'COMMITTEE'

	UNION
		SELECT
			a.ID AS MemberKey,
			a.PRODUCT_CODE AS GroupKey,
			a.DESCRIPTION AS GroupName,
			'Event' AS GroupType,
			'Attendee' AS ROLE,
			NULL AS SinceDate,
			NULL AS StartDate,
			NULL AS EndDate
		FROM Activity a
			LEFT OUTER JOIN Meet_Master b ON a.PRODUCT_CODE = b.MEETING
		WHERE 
		a.ACTIVITY_TYPE = 'MEETING'

	UNION
		SELECT
			a.ST_ID AS MemberKey,
			b.MEETING AS GroupKey,
			c.TITLE AS GroupName,
			'Event' AS GroupType,
			CASE
			WHEN c.BEGIN_DATE > GetDate() THEN 'Registrant'
			ELSE 'Attendee'
		END AS ROLE,
			NULL AS SinceDate,
			NULL AS StartDate,
			NULL AS EndDate
		FROM Orders a
			INNER JOIN Order_Meet b ON a.ORDER_NUMBER = b.ORDER_NUMBER
			LEFT OUTER JOIN Meet_Master c ON b.MEETING = c.MEETING
		WHERE 
		a.STAGE <> 'CANCELED'
			AND a.Status <> 'C' 
GO
/*
	Author: Higher Logic
	Purpose: Creates security group memberships for use in the Higher Logic platform.
	Remarks: Users are added to security groups in HL based ON the GroupKey, so it is important that the GroupKey is unique for each intended security group.
*/
CREATE VIEW dbo.hlMemberSecurityGroups
AS
	SELECT
		a.ID AS MemberKey,
		a.MEMBER_TYPE AS GroupKey,
		b.DESCRIPTION AS GroupName,
		'Member' AS GroupType,
		'Member' AS ROLE,
		a.JOIN_DATE AS SinceDate,
		NULL AS StartDate,
		a.PAID_THRU AS EndDate
	FROM Name a
		INNER JOIN Member_Types b
		ON a.MEMBER_TYPE = b.MEMBER_TYPE
GO
/*
	Author: Higher Logic
	Purpose: Returns member details for a given MemberKey.
	Remarks: Modify the values returned for each item as needed. Note that Member_Types.COMPANY_RECORD is used to determine if a record is a company record.
*/
CREATE PROCEDURE dbo.hlGetMemberDetails
	@MemberKey varchar(10)
AS
BEGIN
	SELECT
		n.ID AS MemberKey,
		n.Prefix,
		n.First_Name AS FirstName,
		n.Last_Name AS LastName,
		n.Suffix,
		n.Designation,
		n.Informal AS InformalName,
		n.Company AS CompanyName,
		n.Title,
		n.Email AS Email1,
		'' AS Email2,
		Work_Phone AS Phone1,
		'Work' AS Phone1Type,
		n.Home_Phone AS Phone2,
		'Home' AS Phone2Type,
		n.Fax AS Phone3,
		'Fax' AS Phone3Type,
		n.Toll_Free AS Phone4,
		'TollFree' AS Phone4Type,
		n.Website AS WebsiteUrl,
		mt.COMPANY_RECORD AS IsCompany,
		n.CO_ID AS CompanyID,
		n.Status,
		n.EXCLUDE_DIRECTORY AS DoNotList,
		CAST(0 AS bit) AS DoNotEmail,
		'' AS PictureUrl,
		'' AS Custom01,
		'' AS Custom02,
		'' AS Custom03,
		'' AS Custom04,
		'' AS Custom05,
		'' AS Custom06,
		'' AS Custom07,
		'' AS Custom08,
		'' AS Custom09,
		'' AS Custom10
	FROM Name n
		INNER JOIN Member_Types mt ON n.MEMBER_TYPE = mt.MEMBER_TYPE
	WHERE n.ID = @MemberKey
END
GO
/*
	Author: Higher Logic
	Purpose: Returns member address details for a given MemberKey.
	Remarks: Modify the values returned for each item as needed.
*/
CREATE PROCEDURE dbo.hlGetMemberAddresses
	@MemberKey varchar(10)
AS
BEGIN
	SELECT
		a.PURPOSE AS AddressType,
		a.Address_1 AS Address1,
		a.Address_2 AS Address2,
		a.Address_3 AS Address3,
		a.City,
		a.State_Province AS StateProvince,
		a.ZIP AS PostalCode,
		a.Country AS CountryCode,
		a.PHONE AS Phone,
		a.FAX AS Fax,
		a.TOLL_FREE AS TollFree,
		a.PREFERRED_MAIL AS IsPreferredMail,
		a.PREFERRED_BILL AS IsPreferredBill,
		1 AS IsPreferredList,
		1 AS IsListCityState,
		a.EMAIL AS EmailAddress,
		CASE a.Address_Num
			WHEN n.Address_Num_1 THEN 1
			WHEN n.Address_Num_2 THEN 2
			WHEN n.Address_Num_3 THEN 3
		END AS AddressNumber,
		CAST(a.ADDRESS_NUM AS nvarchar(10)) AS LegacyAddressKey
	FROM Name n
		INNER JOIN Name_Address a
		ON n.Address_Num_1 = a.Address_Num
			OR n.Address_Num_2 = a.Address_Num
			OR n.Address_Num_3 = a.Address_Num
	WHERE n.ID = @MemberKey
	ORDER BY AddressNumber
END
GO
/*
	Author: Higher Logic
	Purpose: Returns community memberships for a given MemberKey.
*/
CREATE PROCEDURE dbo.hlGetMemberCommunityGroups
	@MemberKey varchar(10)
AS
BEGIN
	SELECT
		GroupKey,
		GroupName,
		GroupType,
		ROLE,
		SinceDate,
		StartDate,
		EndDate
	FROM hlMemberCommunityGroups
	WHERE MemberKey = @MemberKey
END 
GO
/*
	Author: Higher Logic
	Purpose: Returns community memberships for a given GroupKey.
*/
CREATE PROCEDURE dbo.hlGetCommunityGroupMembers
	@GroupKey varchar(100),
	@GroupType varchar(100)
AS
BEGIN
	SELECT
		MemberKey,
		ROLE,
		SinceDate,
		StartDate,
		EndDate
	FROM hlMemberCommunityGroups
	WHERE GroupKey = @GroupKey
END 
GO
/*
	Author: Higher Logic
	Purpose: Returns security group memberships for a given GroupKey.
*/
CREATE PROCEDURE dbo.hlGetSecurityGroupMembers
	@GroupKey varchar(100)
AS
BEGIN
	SELECT
		MemberKey,
		ROLE,
		SinceDate,
		StartDate,
		EndDate
	FROM hlMemberSecurityGroups
	WHERE GroupKey = @GroupKey
END 
GO
/*
	Author: Higher Logic
	Purpose: Returns security group GroupKeys.
*/
CREATE PROCEDURE dbo.hlGetSecurityGroups
AS
BEGIN
	SELECT
		GroupKey,
		count(*) AS COUNT
	FROM hlMemberSecurityGroups
	GROUP BY GroupKey
END 
GO
/*
	Author: Higher Logic
	Purpose: Returns security group memberships for a given MemberKey.
*/
CREATE PROCEDURE dbo.hlGetMemberSecurityGroups
	@MemberKey varchar(10)
AS
BEGIN
	SELECT
		GroupKey,
		GroupName,
		GroupType,
		ROLE,
		SinceDate,
		StartDate,
		EndDate
	FROM hlMemberSecurityGroups
	WHERE MemberKey = @MemberKey
END 
GO
/*
	Author: Higher Logic
	Purpose: Returns MemberKeys of changed records to be refreshed in Higher Logic.
	Remarks: Default behavior looks at Name.LAST_UPDATED, Activity.TRANSACTION_DATE, Orders.UPDATED_DATE_TIME, Orders.ENTERED_DATE_TIME.
	         Since Activity records *could* be updated depending on use, it might make sense to put a trigger on update of Activity and use UF_7 to store an updated date.
             Extend this to add checks to other criteria.
*/
CREATE PROCEDURE [dbo].[hlGetChangedMembers]
	@ChangedSince datetime
AS
BEGIN
	DECLARE @ChangedRecords TABLE 
	(
		MemberKey varchar(10),
		LastUpdated datetime
	)

	INSERT INTO @ChangedRecords
							SELECT
			ID AS MemberKey,
			LAST_UPDATED AS LastUpdated
		FROM Name
		WHERE LAST_UPDATED >= @ChangedSince

	UNION
		---- If there are transaction dates with a time value of 00:00:00.000, then use the SELECT statement below:
		SELECT
			a.ID AS MemberKey,
			MAX(COALESCE(a.TRANSACTION_DATE, '2000-12-02 00:00:00:000')) AS LastUpdated
		FROM Activity a
			INNER JOIN Name b ON a.ID = b.ID
		WHERE 
			CAST(a.TRANSACTION_DATE AS DATE) >= CAST(@ChangedSince AS DATE)
			AND NOT a.TRANSACTION_DATE > DATEADD(D,1, GetDate())
		GROUP BY a.ID
	---- If there aren't transaction dates with time value of 00:00:00.000, then delete the SELECT statement above and use the SELECT statement below:
	--SELECT 
	--a.ID AS MemberKey, 
	--MAX(COALESCE(a.TRANSACTION_DATE, '2000-12-02 00:00:00:000')) AS LastUpdated
	--FROM Activity a
	--INNER JOIN Name b ON a.ID = b.ID
	--WHERE a.TRANSACTION_DATE >= @ChangedSince
	--GROUP BY a.ID

	UNION
		SELECT
			a.ST_ID AS MemberKey,
			MAX(COALESCE(a.UPDATED_DATE_TIME, '2000-12-02 00:00:00:000')) AS LastUpdated
		FROM Orders a
			INNER JOIN Name b ON a.ST_ID = b.ID
		WHERE a.UPDATED_DATE_TIME >= @ChangedSince
		GROUP BY a.ST_ID

	UNION
		SELECT
			a.BT_ID AS MemberKey,
			MAX(COALESCE(a.UPDATED_DATE_TIME, '2000-12-02 00:00:00:000')) AS LastUpdated
		FROM Orders a
			INNER JOIN Name b ON a.BT_ID = b.ID
		WHERE 
			a.UPDATED_DATE_TIME >= @ChangedSince
			AND a.BT_ID <> a.ST_ID
		GROUP BY a.BT_ID

	UNION
		SELECT
			a.ST_ID AS MemberKey,
			MAX(COALESCE(a.ENTERED_DATE_TIME, '2000-12-02 00:00:00:000')) AS LastUpdated
		FROM Orders a
			INNER JOIN Name b ON a.ST_ID = b.ID
		WHERE a.ENTERED_DATE_TIME >= @ChangedSince
		GROUP BY a.ST_ID

	UNION
		SELECT
			a.BT_ID AS MemberKey,
			MAX(COALESCE(a.ENTERED_DATE_TIME, '2000-12-02 00:00:00:000')) AS LastUpdated
		FROM Orders a
			INNER JOIN Name b ON a.BT_ID = b.ID
		WHERE 
			a.ENTERED_DATE_TIME >= @ChangedSince
			AND a.BT_ID <> a.ST_ID
		GROUP BY a.BT_ID

	SELECT
		a.MemberKey,
		(CASE
			WHEN MAX(a.LastUpdated) > GetDate() 
			THEN GetDate()
			ELSE MAX(a.LastUpdated)
		END) AS LastUpdated
	FROM @ChangedRecords a
		INNER JOIN Name b ON a.MemberKey = b.ID
	WHERE 
	(
		EXISTS
		(
			SELECT 1
			FROM Member_Types c
			WHERE 
			(
				b.MEMBER_TYPE = c.MEMBER_TYPE
				OR b.PREVIOUS_MT = c.MEMBER_TYPE
			)
		)
		OR 
		EXISTS
		(
			SELECT 1
			FROM Orders c
				INNER JOIN ORDER_MEET d ON c.ORDER_NUMBER = d.ORDER_NUMBER
			WHERE 
			(
				b.ID = c.ST_ID
				OR b.ID = c.BT_ID
			)
			AND c.ORDER_DATE > GetDate() - 730
		)
	)
	AND
	(
		a.LastUpdated > GetDate() - 180
	)
	GROUP BY a.MemberKey
	-- If Name_Log.SUB_TYPE = 'delete' is NOT used, then use the ORDER BY below:
	ORDER BY MAX(a.LastUpdated)
-- If Name_Log.SUB_TYPE = 'delete' is used, delete the ORDER BY above and add the SELECT statement below:
--	UNION
--		SELECT 
--			ID AS MemberKey, 
--			MAX(Date_time) AS LastUpdated
--		FROM Name_Log
--		WHERE SUB_TYPE = 'delete'
--			AND DATE_TIME >= @ChangedSince
--		GROUP BY ID
--		ORDER BY LastUpdated
END
GO

CREATE PROCEDURE [dbo].[hlWriteActivity] @MemberKey varchar(10), @ActivityType varchar(10), @ActivityDate datetime, @Description varchar(255), @seqn int out  AS 
BEGIN   
  DECLARE @error int
  DECLARE @rowcount int
  if @ActivityType <> '' AND exists(select Prod_Type from Product_Type where Prod_Type = @ActivityType)
  BEGIN
    exec dbo.sp_asi_GetCounter2 'Activity', @seqn OUT

      INSERT INTO dbo.ACTIVITY (SEQN, ID, ACTIVITY_TYPE, TRANSACTION_DATE, DESCRIPTION)
      VALUES (@seqn, @MemberKey, @ActivityType, @ActivityDate, @Description)
      select @error = @@ERROR , @rowcount = @@ROWCOUNT 
      if @error <> 0 OR @rowcount < 1
      BEGIN
         --insert failed
      
         set @seqn = 0
      END
      
  END
  else
  BEGIN
   --activity type is blank or not valid for customer
    set @seqn = 0
  END
END
GO